﻿# Type this to run the command in a Powershell session  
# .\RunMigration.ps1 -ConfigFile 'Customer.config' -TestMode 'true'
# These are command line parameters that can be set on each run or they can be changed here
param ($ConfigFile = '', $SQLServername = '', $CommonDatabaseName = '', $EmailTo = ''
    , $TestMode = 'false', $QuietMode = 'quiet', $Path = '')
function LogMessage  { [CmdletBinding()] Param (  [Parameter(Mandatory=$true, Position=0)] [string]$LogMessage, [Parameter(Position=1)] [string]$Color )
    if ($Color -eq '') {Write-Output ("{0} - {1}" -f (Get-Date -Format "MM/dd/yyyy hh:mm:ss tt"), $LogMessage) }
    else {Write-Host  -BackgroundColor $Color ("{0} - {1}" -f (Get-Date -Format "MM/dd/yyyy hh:mm:ss tt"), $LogMessage) }
}
    $StartTime = $(get-date)
    LogMessage "Config file name is: $Path$ConfigFile"
    if ( ![string]::IsNullOrEmpty( $Path ) ){ Set-Location -Path "$Path"}
    # This is where the NoUAC.Manager.exe is located.  D:\Aprima\.PRMBuild\BuildDeployment\Installation\NoUAC.Manager.exe
    $Manager = 'D:\Aprima\.PRMBuild\BuildDeployment\Installation\NoUAC.Manager.exe'
    LogMessage "Running Manager from: $Manager `n"

	if ($TestMode -eq 'true')	{LogMessage "*** TestMode is on ***"}
    
	## cast the config files to XML objects
	[xml]$Customer = Get-Content -Path $ConfigFile
    if (Test-Path "Email.config") {
        [xml]$Email = Get-Content -Path "Email.config" 
        $EmailTo = $Email.configuration.emailto.address
     }

## Database migration
	## if the param is empty, override from the config
	$DatabaseName = $Customer.configuration.database.name
	if ( [string]::IsNullOrEmpty( $SQLServername ) ){$SQLServername = $Customer.configuration.database.server}
	if ( [string]::IsNullOrEmpty( $CommonDatabaseName ) ){$CommonDatabaseName = $Customer.configuration.database.common}
    $BackupLocation = $Customer.configuration.database.backupfolder
    $BackUpDatabase = ![string]::IsNullOrEmpty( $BackupLocation )
	
	$AnswerFilePath = Join-Path -Path  $pwd.Path -ChildPath ($DatabaseName + "-Migration.xml")
	$LogFilePath = $env:ProgramData + "\Aprima\InstallationManager\" + $env:USERNAME +"\Manager-"+ $DatabaseName +".log"
	# this is the AnswerFile template for Migration
	$AnswerFile = "<InstallationAnswers>
		<InstallAction Mode=""$QuietMode""/>
		<SqlConnection ServerName=""$SQLServername""/>
		<Migration CommonDatabaseName=""$CommonDatabaseName"" />
		<SendEmail SendTo=""$EmailTo"" Status=""$DatabaseName migration""/>
        <BackupDatabase BackUpDatabase=""$BackUpDatabase"" BackUpLocation=""$BackupLocation"" />
	</InstallationAnswers>"

	#save the AnswerFile contents
	Set-Content -path $AnswerFilePath -Value $AnswerFile
	LogMessage "Migrating Database: $DatabaseName on SQLServer $SQLServername `n  LogFile $LogFilePath`n  AnswerFile $AnswerFilePath"
	if ($TestMode -eq 'false')	{
		# use the next line to watch the database migration log file
		# Start-Process -FilePath "powershell.exe"  "-command Get-Content $LogFilePath -Tail 10 -Wait"
		# use the next 2 lines to watch the main manager log file
		# $ManagerLogFilePath = $env:ProgramData + "\Aprima\InstallationManager\" + $env:USERNAME +"\Manager.log"
		# Start-Process -FilePath "powershell.exe"  "-command Get-Content $ManagerLogFilePath -Tail 10 -Wait"
		# & $Manager $AnswerFilePath Migration $DatabaseName  | Out-Null
		
		$process = start-process $Manager -ArgumentList "$AnswerFilePath Migration $DatabaseName" -PassThru -Wait
		# LogMessage "ExitCode: $process.ExitCode"
		if ($process.ExitCode -ne 0) { throw "$DatabaseName - migration failed. Check the log file for errors." } 
        else { LogMessage "$DatabaseName - migration finished.`n" Green} 
	}
	else	{ LogMessage "Database $DatabaseName Answerfile: $AnswerFile" }


## Weekly Updates
	if ($TestMode -eq 'false')	{
        $QueryPRM = "select CONCAT(PRMVersion, '.' ,PRMSeq) [version] from VersionPRM where Status = 0"
        $DBresult = invoke-sqlcmd -ServerInstance $SQLServername -Database $DatabaseName -Query $QueryPRM
        $DBVersion = $DBresult.version
        Write-Host "`nRunning Weekly Updates Version: $DBVersion Database: $DatabaseName SQLServer: $SQLServername"
		$process = start-process $Manager -ArgumentList "runupdatesconsole $SQLServername $DatabaseName $DBVersion" -PassThru -Wait
		if ($process.ExitCode -ne 0) { throw "$DatabaseName - Weekly Updates failed. Check the log file for errors." } 
        else { LogMessage "$DatabaseName - Weekly Updates finished.`n" Green} 
	}



## Appserver upgrade
	$Customer.configuration.appservers.service	 |  ForEach-Object {
		$AppServerName = $_.server
		$AppserviceName = $_.name
		$AppservicePath = $_.path
		
		$AnswerFilePath = Join-Path -Path  $pwd.Path -ChildPath ($AppserviceName + ".xml")
		$LogFilePath = $env:ProgramData + "\Aprima\InstallationManager\" + $env:USERNAME +"\"+ $AppserviceName +".log"
		# this is the AnswerFile template for Appserver
		$AnswerFile = "<InstallationAnswers>
			<InstallAction Mode=""$QuietMode""/>
			<AppServer AppServerAction=""Update""
				AppServiceName=""$AppserviceName"" 
				AppServerName=""$AppServerName""
				AppServerLocation=""$AppservicePath""
				/>
			<SendEmail SendTo=""$EmailTo"" SendFailed=""true"" Status=""$DatabaseName Appserver""/>
			</InstallationAnswers>"
		#save the AnswerFile contents
		Set-Content -path $AnswerFilePath -Value $AnswerFile
		LogMessage "Upgrade Appserver: $AppserviceName on $AppServerName `n  LogFile: $LogFilePath `n  AnswerFile: $AnswerFilePath"
		if ($TestMode -eq 'false') {
			# use the next line to watch the log file
			# Start-Process -FilePath "Powershell.exe"  "-command Get-Content $LogFilePath -Tail 10 -Wait"
			# use the next 2 lines to watch the main manager log file
			# $ManagerLogFilePath = $ENV:ProgramData + "\Aprima\InstallationManager\" + $ENV:USERNAME +"\Manager.log"
			# Start-Process -FilePath "Powershell.exe"  "-command Get-Content $ManagerLogFilePath -Tail 10 -Wait"
			# & $Manager $AnswerFilePath AppServer

			$process = start-process $Manager -ArgumentList "$AnswerFilePath AppServer $DatabaseName-AppServer" -PassThru -Wait
			# LogMessage "ExitCode: $process.ExitCode"
			if ($process.ExitCode -ne 0) { throw "$AppserviceName - upgrade failed. Check the log file for errors." } 
            else {LogMessage "$AppserviceName - upgrade finished.`n" Green; 
                Add-Content -Path "$AppservicePath\Manager.log"  -Value ("{0} - {1}" -f (Get-Date -Format "MM/dd/yyyy hh:mm:ss tt"), "$AppserviceName - upgrade finished.") -ErrorAction SilentlyContinue }
		
		}
		else	{ LogMessage "Appserver Answerfile: $AnswerFile" }
	}
    # send a finished email for the migration
    if($Error.Count -eq 0)  {$FinalMessage = "- upgrade completed successfully"} else { $FinalMessage = "- upgrade failed - check log files for errors"}
    if ($TestMode -eq 'false') {$process = start-process $Manager -ArgumentList "sendemail ""$EmailTo"" $DatabaseName $FinalMessage" -PassThru -Wait }
    $totalTime = "{0:HH:mm:ss}" -f ([datetime]($(get-date) - $StartTime).Ticks)
    LogMessage "$DatabaseName upgrade complete - elapsed time $totalTime" Green