(function (window) {
    'use strict';

    // Adapted from https://github.com/crdeutsch/MonoTouch-JsBridge used on IoS
    // most of this library is borrowed from Titanium, Copyright 2008-2012 Appcelerator, Inc. under the Apache License Version 2 http://www.apache.org/licenses/LICENSE-2.0
    window.bridge = {};
    bridge.appId = 'jsbridge';
    bridge.pageToken = 'index';
    bridge.app = {};
    bridge.api = {};
    bridge.app._listeners = {};
    bridge.app._listener_id = 1;
    bridge.app.id = bridge.appId;

    //used for events fired before the devicetype is setup.
    bridge.eventQueue = [];

    bridge._broker = function (module, method, data) {
        var deviceBroker = trySetDeviceBroker(),
            url = 'app://' + module + '/' + method + '?data=' + encodeURIComponent(JSON.stringify(data)) + '&_=' + Math.random();

        if (window.bridgeid) {
            url += '&bridgeid=' + window.bridgeid;
        }

        if (deviceBroker) {
            deviceBroker(url);
        }
        else {
            window.console.info('Bridge not setup (missing devicetype). Event queued until bridge is configured.');

            bridge.eventQueue.push({
                url : url
            });
        }
    };

    function trySetDeviceBroker(){
        if (!bridge._deviceBroker) {
            switch (window.devicetype) {
                case 'ios':
                    window.console.info('Using iOS broker in JsBridge');
                    bridge._deviceBroker = bridge._iosBroker;
                    break;
                case 'android':
                    window.console.info('Using Android broker in JsBridge');
                    bridge._deviceBroker = bridge._androidBroker;
                    break;
                case 'prm':
                    window.console.info('Using PRM broker in JsBridge');
                    bridge._deviceBroker = bridge._prmBroker;
                    break;
                default:
                    window.console.info('Unknown bridge type, broker left undefined.');
                    break;
            }

            //if the device broker was set, we need to clear the queue
            if (bridge._deviceBroker) {
                window.console.info('Bridge broker has been set. Processing events from the queue: ' + bridge.eventQueue.length);

                while (bridge.eventQueue.length) {
                    var event = bridge.eventQueue.shift(),
                        url = event.url;

                    if (window.bridgeid) {
                        url += '&bridgeid=' + window.bridgeid;
                    }

                    bridge._deviceBroker(url);
                }
                delete bridge.eventQueue;
            }
        }

        return bridge._deviceBroker;
    }

    bridge.app._dispatchEvent = function (type, evt) {
        var listeners = bridge.app._listeners[type];
        if (listeners) {
            for (var c = 0; c < listeners.length; c++) {
                var entry = listeners[c];
                entry.callback.call(entry.callback, evt);
            }
        }
    };
    bridge.app.fireEvent = function (name, evt) {
        bridge._broker('App', 'fireEvent', { name: name, event: evt });
    };

    bridge.app.addEventListener = function (name, fn) {
        var listeners = bridge.app._listeners[name];
        if (typeof (listeners) == 'undefined') {
            listeners = [];
            bridge.app._listeners[name] = listeners;
        }
        var newid = bridge.pageToken + bridge.app._listener_id++;
        listeners.push({ callback: fn, id: newid });
    };

    bridge.app.removeEventListener = function (name, fn) {
        var listeners = bridge.app._listeners[name];
        if (listeners) {
            for (var c = 0; c < listeners.length; c++) {
                var entry = listeners[c];
                if (entry.callback == fn) {
                    listeners.splice(c, 1);
                    break;
                }
            }
        }
    };

    function setupXhr() {
        // jXHR library
        (function (global) {
            var setTimeout = global.setTimeout,
                doc = global.document,
                callbackCounter = 0;

            global.CustomXhr = function () {
                var scriptUrl,
                    scriptLoaded,
                    scriptElem,
                    publicApi = null;

                function removeScript() {
                    try {
                        scriptElem.parentNode.removeChild(scriptElem);
                    } catch (err) {
                    }
                }

                function reset() {
                    scriptLoaded = false;
                    scriptUrl = '';
                    removeScript();
                    scriptElem = null;
                    fireReadyStateChange(0);
                }

                function throwError(msg) {
                    try {
                        publicApi.onerror.call(publicApi, msg, scriptUrl);
                    } catch (err) {
                        throw new Error(msg);
                    }
                }

                function handleScriptLoad() {
                    if ((this.readyState && this.readyState !== 'complete' && this.readyState !== 'loaded') || scriptLoaded) {
                        return;
                    }
                    this.onload = this.onreadystatechange = null; // prevent memory leak
                    scriptLoaded = true;
                    if (publicApi.readyState !== 4) throwError('Script failed to load [' + scriptUrl + '].');
                    removeScript();
                }

                function fireReadyStateChange(rs, args) {
                    args = args || [];
                    publicApi.readyState = rs;
                    if (typeof publicApi.onreadystatechange === 'function') publicApi.onreadystatechange.apply(publicApi, args);
                }

                publicApi = {
                    onerror: null,
                    onreadystatechange: null,
                    readyState: 0,
                    open: function (method, url) {
                        reset();
                        var internalCallback = 'cb' + (callbackCounter++);
                        (function (icb) {
                            global.CustomXhr[icb] = function () {
                                try {
                                    fireReadyStateChange.call(publicApi, 4, arguments);
                                }
                                catch (err) {
                                    publicApi.readyState = -1;
                                    throwError('Script failed to run [' + scriptUrl + '].');
                                }
                                global.CustomXhr[icb] = null;
                            };
                        })(internalCallback);
                        scriptUrl = url.replace(/=\?/, '=jXHR.' + internalCallback);
                        fireReadyStateChange(1);
                    },
                    send: function () {
                        setTimeout(function () {
                            scriptElem = doc.createElement('script');
                            scriptElem.setAttribute('type', 'text/javascript');
                            scriptElem.onload = scriptElem.onreadystatechange = function () {
                                handleScriptLoad.call(scriptElem);
                            };
                            scriptElem.setAttribute('src', scriptUrl);
                            doc.getElementsByTagName('head')[0].appendChild(scriptElem);
                        }, 0);
                        fireReadyStateChange(2);
                    },
                    setRequestHeader: function () {
                    },
                    getResponseHeader: function () {
                        return '';
                    },
                    getAllResponseHeaders: function () {
                        return [];
                    }
                };

                reset();

                return publicApi;
            };
        })(window);
    }


    bridge._iosBroker = function (initialUrl){
        //setup xhr first time this broker is called, and then replace this broker with the configured broker.
        setupXhr();
        bridge.app.CustomXhr = CustomXhr;

        bridge._iosBroker = function (url) {
            var x1 = new bridge.app.CustomXhr();
            x1.onerror = function (e) {
                console.log('XHR error:' + JSON.stringify(e));
            };
            x1.open('GET', url);
            x1.send();
        };

        return bridge._iosBroker(initialUrl);
    };

    bridge._androidBroker = function (url){
        var result = window.AndrApp.Event(url);
        result = JSON.parse(result);

        if (!result.Success) {
            console.log('Android bridge error: ' + JSON.stringify(result));
        }
    };

    bridge._prmBroker = function (url) {
        var result = window.external.HandleEvent(url);
        result = JSON.parse(result);

        if (!result.Success) {
            console.log('PRM bridge error: ' + JSON.stringify(result));
        }
    };

})(window);
